/*
 * Pipelined MIPS processor with JR and JAL
 * (c) 2008 The University of Aizu
 */

module mipspipe
(
    clk, reset,
 
    dmem_data,
    dmem_write,
    dmem_read,
    dmem_adr,
    dmem_data_out,
 
    imem_data,
    imem_adr
 );

   //input and output
   input         clk, reset;
 
   input [31:0]  dmem_data;
   output 	 dmem_write;
   output 	 dmem_read;
   output [31:0] dmem_adr;
   output [31:0] dmem_data_out;
 
   input [31:0]  imem_data;
   output [31:0] imem_adr;
   
   // Internal wires for pipeline register
   wire [31:0] if_pcinc, if_curpc, if_nextpc, if_imem_data;
   
   wire [31:0] id_pcadr, id_inst, ex_inst, id_pc_src, id_data_rega, id_data_regb;
   wire [31:0] id_rf_outA, id_rf_outB, id_jmp_adr, id_bch_adr, id_data_imm;
   wire [3:0]  id_ctrl_ex;
   wire [1:0]  id_ctrl_mem, id_aluop;
   wire [2:0]  id_ctrl_wb;
   wire [8:0]  id_ctrl_out;
   wire [1:0]  id_fwd_A, id_fwd_B;
   wire        id_reg_dst, id_alusrc, id_branch, id_jump, id_memread;
   wire        id_memwrite, id_regwrite, id_memtoreg, id_rawrite;
   wire        id_data_eq, id_jr, id_bch_taken, id_pcdst, id_stall, id_flush;
   wire        id_fwd_ex_id_A, id_fwd_mem_id_A, id_fwd_ex_id_B, id_fwd_mem_id_B;
   wire        id_fwd_dmem_id_A, id_fwd_dmem_id_B;



   wire [31:0] ex_data_rega, ex_data_regb, ex_data_imm, ex_alusrcb;
   wire [31:0] ex_data_out, ex_pcadr, ex_data_alures;
   wire [4:0]  ex_wbdst_imm, ex_wbdst_reg, ex_wbdst;
   wire [3:0]  ex_ctrl_ex;
   wire [1:0]  ex_ctrl_mem, ex_aluop;
   wire [2:0]  ex_ctrl_wb, ex_aluc;
   wire        ex_alusrc, ex_reg_dst;

   wire [31:0] mem_data_out, mem_data_alures, mem_pcadr;   
   wire [1:0]  mem_ctrl_mem;   
   wire [2:0]  mem_ctrl_wb;
   wire [4:0]  mem_wbdst;
   wire        mem_memwrite, mem_memread;

   wire [31:0] wb_data_alures, wb_data_mem, wb_regdata, wb_pcadr;
   wire [2:0]  wb_ctrl_wb;
   wire [4:0]  wb_wbdst, wb_regdst;
   wire        wb_memtoreg, wb_regwrite, wb_rawrite;

   
   // Output port connection
   assign  imem_adr      = if_curpc;
   assign  dmem_write    = mem_memwrite;
   assign  dmem_read     = mem_memread;
   assign  dmem_adr      = mem_data_alures;
   assign  dmem_data_out = mem_data_out;
   
   ///////////////////////////////////////////
   // 		    IF stage		    //
   ///////////////////////////////////////////
   register_re if_pc
     (
      .clock (clk),
      .reset (reset),
      .en    (id_stall),
      .in    (if_nextpc),
      .out   (if_curpc)
      );
   
   adder if_incadder
     (
      .inA (if_curpc),
      .inB (32'h00000004),
      .out (if_pcinc)
      );
   
   mux2 if_mux_pcsource
     (
      .in0 (if_pcinc),
      .in1 (id_pc_src),
      .sel (id_pcdst),
      .out (if_nextpc)
      );
   
   mux2 if_imem_data_mux
     (
      .in0 (imem_data),
      .in1 (`NOP),
      .sel (id_flush),
      .out (if_imem_data)
      );
   
   // Pipeline registers from IF stage to ID stage
   register ifid_adr_current
     (
      .clock (clk),
      .in    (if_pcinc),
      .out   (id_pcadr)
      );
   
   register_e ifid_data_inst
     (
      .clock (clk),
      .en    (id_stall),
      .in    (if_imem_data),
      .out   (id_inst)
      );

   ///////////////////////////////////////////
   // 		    ID stage		    //
   ///////////////////////////////////////////
   assign  id_ctrl_ex   = id_ctrl_out[8:5];
   assign  id_ctrl_mem  = id_ctrl_out[4:3];
   assign  id_ctrl_wb   = id_ctrl_out[2:0];
   assign  id_jmp_adr   = {id_pcadr[29:26], id_inst[25:0], 2'b00};
   assign  id_data_eq   = (id_rf_outA == id_rf_outB);
   assign  id_bch_taken = id_branch & id_data_eq;
   assign  id_jr        = (id_inst[31:26] == `MIPSINST_R_TYPE) &
			  (id_inst[5:0] == `MIPSFUNC_JR);
   assign  id_pcdst     = id_jump | id_bch_taken | id_jr;
   assign  id_flush     = reset | id_pcdst;
   assign  id_fwd_A     = (id_fwd_dmem_id_A) ? 2'b11 :
			  (id_fwd_mem_id_A)  ? 2'b10 :
			  (id_fwd_ex_id_A)   ? 2'b01 : 2'b00;
   assign  id_fwd_B     = (id_fwd_dmem_id_B) ? 2'b11 :
			  (id_fwd_mem_id_B)  ? 2'b10 :
			  (id_fwd_ex_id_B)   ? 2'b01 : 2'b00;
			  
   
   hazard id_hzd_detection
     (
      .id_source    (id_inst[25:21]),
      .id_target    (id_inst[20:16]),
      .ex_mem_read  (ex_ctrl_mem[1]),
      .ex_dst       (ex_wbdst),
      .mem_mem_read (mem_memread),
      .mem_dst      (mem_wbdst),
      .id_inst_jr   (id_jr),
      .mem_rawrite  (mem_ctrl_wb[0]),
      .stall        (id_stall)
      );
   
   forwarding id_fwd_unit
     (
      .id_source     (id_inst[25:21]),
      .id_target     (id_inst[20:16]),
      .ex_dst        (ex_wbdst),
      .ex_write      (ex_ctrl_wb[2]),
      .mem_dst       (mem_wbdst),
      .mem_write     (mem_ctrl_wb[2]),
      .mem_memread   (mem_memread),
      .fwd_ex_id_A   (id_fwd_ex_id_A),
      .fwd_mem_id_A  (id_fwd_mem_id_A),
      .fwd_dmem_id_A (id_fwd_dmem_id_A),
      .fwd_ex_id_B   (id_fwd_ex_id_B),
      .fwd_mem_id_B  (id_fwd_mem_id_B),
      .fwd_dmem_id_B (id_fwd_dmem_id_B)
      );
   
   regfile id_regfile
     (
      .clk         (clk),
      .write       (wb_regwrite),
      .adr_wport   (wb_regdst),
      .data_wport  (wb_regdata),
      .adr_rporta  (id_inst[25:21]),
      .adr_rportb  (id_inst[20:16]),
      .data_rporta (id_data_rega),
      .data_rportb (id_data_regb)
      );
   
   instdec_pipe id_inst_decoder
     (
      .Op       (id_inst[31:26]),
      .Branch   (id_branch),
      .Jump     (id_jump),
      .RegDst   (id_reg_dst),
      .ALUOp    (id_aluop),
      .ALUSrc   (id_alusrc),
      .MemRead  (id_memread),
      .MemWrite (id_memwrite),
      .RegWrite (id_regwrite),
      .MemtoReg (id_memtoreg),
      .RAWrite  (id_rawrite),
      .ExtType  (id_ext_type)
      );
   
   ext id_extend
     (
      .extendType (id_ext_type),
      .in         (id_inst[15:0]),
      .out        (id_data_imm)
      );

   mux4 id_mux_rf_outA
     (
      .in0 (id_data_rega),
      .in1 (ex_data_alures),
      .in2 (mem_data_alures),
      .in3 (dmem_data),
      .sel (id_fwd_A),
      .out (id_rf_outA)
      );
   
   mux4 id_mux_rf_outB
     (
      .in0 (id_data_regb),
      .in1 (ex_data_alures),
      .in2 (mem_data_alures),
      .in3 (dmem_data),
      .sel (id_fwd_B),
      .out (id_rf_outB)
      );
   
   adder id_bch_calc
     (
      .inA (id_pcadr),
      .inB ({id_data_imm[29:0],2'b00}),
      .out (id_bch_adr)
      );
   
   mux4 id_mux_pcsrc 
     (
      .in0 (id_rf_outA),
      .in1 (id_jmp_adr),
      .in2 (id_bch_adr),
      .in3 (32'bz),
      .sel ({id_branch,id_jump}),
      .out (id_pc_src)
      );

   mux2 id_ctrl_mux
     (
      .in0 ({id_reg_dst, id_aluop, id_alusrc, id_memread,
	     id_memwrite, id_regwrite, id_memtoreg, id_rawrite}),
      .in1 (9'b0),
      .sel (id_stall),
      .out (id_ctrl_out)
      );
   
   // Pipeline registers from ID stage to EX stage
   register_r idex_inst
     (
      .clock (clk),
      .reset (id_stall),
      .in    (id_inst),
      .out   (ex_inst)
      );
   
   register idex_ctrl_ex
     (
      .clock (clk),
      .in    (id_ctrl_ex),
      .out   (ex_ctrl_ex)
      );
   
   register_r idex_ctrl_mem
     (
      .clock (clk),
      .reset (reset),
      .in    (id_ctrl_mem),
      .out   (ex_ctrl_mem)
      );
   
   register_r idex_ctrl_wb
     (
      .clock (clk),
      .reset (reset),
      .in    (id_ctrl_wb),
      .out   (ex_ctrl_wb)
      );
   
   register idex_adr_current
     (
      .clock (clk),
      .in    (id_pcadr),
      .out   (ex_pcadr)
      );
   
   register idex_data_rega
     (
      .clock (clk),
      .in    (id_rf_outA),
      .out   (ex_data_rega)
      );
   
   register idex_data_regb
     (
      .clock (clk),
      .in    (id_rf_outB),
      .out   (ex_data_regb)
      );
   
   register idex_ext_data
     (
      .clock (clk), 
      .in    (id_data_imm),
      .out   (ex_data_imm)
      );   
   
   ///////////////////////////////////////////
   //  		   EX stage		    //
   ///////////////////////////////////////////
   assign  ex_alusrc  = ex_ctrl_ex[0];
   assign  ex_aluop   = ex_ctrl_ex[2:1];
   assign  ex_reg_dst = ex_ctrl_ex[3];
   
   alucontrol ex_aluctrl
     (
      .field (ex_inst[5:0]),
      .op    (ex_inst[31:26]),
      .aluop (ex_aluop),
      .aluc  (ex_aluc)
      );
   
   mipsalu ex_alu
     (
      .in_a (ex_data_rega),
      .in_b (ex_alusrcb),
      .sel  (ex_aluc),
      .out  (ex_data_alures)
      );
   
   mux2 ex_mux_regdst
     (
      .in0 (ex_inst[20:16]),
      .in1 (ex_inst[15:11]),
      .sel (ex_reg_dst),
      .out (ex_wbdst)
      );
   
   mux2 ex_mux_alusrc
     (
      .in0 (ex_data_regb),
      .in1 (ex_data_imm),
      .sel (ex_alusrc),
      .out (ex_alusrcb)
      );
   
   // Pipeline registers from EX stage to MEM stage
   register_r exmem_ctrl_mem
     (
      .clock (clk),
      .reset (reset),
      .in    (ex_ctrl_mem),
      .out   (mem_ctrl_mem)
      );
   
   register_r exmem_ctrl_wb
     (
      .clock (clk),
      .reset (reset),
      .in    (ex_ctrl_wb),
      .out   (mem_ctrl_wb)
      );
   
   register exmem_data_alures
     (
      .clock (clk),
      .in    (ex_data_alures),
      .out   (mem_data_alures)
      );
   
   register exmem_data_regb
     (
      .clock (clk),
      .in    (ex_data_regb),
      .out   (mem_data_out)
      );
   
   register exmem_wbdst
     (
      .clock (clk),
      .in    (ex_wbdst),
      .out   (mem_wbdst)
      );
   
   register exmem_adr_current
     (
      .clock (clk),
      .in    (ex_pcadr),
      .out   (mem_pcadr)
      );
   
   ///////////////////////////////////////////
   // 		   MEM stage		    //
   ///////////////////////////////////////////
   assign  mem_memwrite = mem_ctrl_mem[0];
   assign  mem_memread  = mem_ctrl_mem[1];
   
   // Pipeline registers from MEM stage to WB stage
   register memwb_ctrl_wb
     (
      .clock (clk),
      .in    (mem_ctrl_wb),
      .out   (wb_ctrl_wb)
      );
   
   register memwb_wbdst
     (
      .clock (clk),
      .in    (mem_wbdst),
      .out   (wb_wbdst)
      );
   
   register memwb_data_alu
     (
      .clock (clk),
      .in    (mem_data_alures),
      .out   (wb_data_alures)
      );
   
   register memwb_data_mem
     (
      .clock (clk),
      .in    (dmem_data),
      .out   (wb_data_mem)
      );
   
   register memwb_adr_current
     (
      .clock (clk),
      .in    (mem_pcadr),
      .out   (wb_pcadr)
      );
   
   ///////////////////////////////////////////
   // 		   WB stage		    //
   ///////////////////////////////////////////
   assign  wb_rawrite  = wb_ctrl_wb[0];
   assign  wb_memtoreg = wb_ctrl_wb[1];
   assign  wb_regwrite = wb_ctrl_wb[2];
   
   mux4 wb_mux_wregdata
     (
      .in0 (wb_data_alures),
      .in1 (wb_data_mem),
      .in2 (wb_pcadr),
      .in3 (32'bz),
      .sel ({wb_rawrite,wb_memtoreg}),
      .out (wb_regdata)
      );
   
   mux2 wb_mux_regdst
     (
      .in0 (wb_wbdst),
      .in1 (5'b11111),
      .sel (wb_rawrite),
      .out (wb_regdst)
      );
   
endmodule
