#include "cal.h"
#include "calcl.h"
#include <string>
#include <iostream>
#include <fstream>

int main(int argc, char** argv)
{
    calInit();

    CALuint numDevices = 0;
    calDeviceGetCount(&numDevices);

    CALdevice device = 0;
    calDeviceOpen(&device, 0);

    CALdeviceattribs attribs;
    attribs.struct_size = sizeof(CALdeviceattribs);
    calDeviceGetAttribs(&attribs, 0);

    CALcontext ctx = 0;
    calCtxCreate(&ctx, device);

    //-------------------------------------------------------------------------
    // Compiling Device Program
    //-------------------------------------------------------------------------
    CALobject obj = NULL;
    CALimage image = NULL;

    CALlanguage lang = CAL_LANGUAGE_IL;
    std::string kernelType = "IL";
    std::string program;

    {
#define LINE_LEN 2048
      std::ifstream asmcode;
      char str[LINE_LEN];
    
      asmcode.open("prog.il");
      if (asmcode.fail()) {
	std::cout << "No input file : \n";
	return -1;
      }

      while(!asmcode.eof()) {
	asmcode.getline(str, LINE_LEN - 1);
	program += str;
	program += "\n";
      }
      asmcode.close();
    }

    if (calclCompile(&obj, lang, program.c_str(), attribs.target) != CAL_RESULT_OK)
    {
        fprintf(stdout, "Program compilation failed. Exiting.\n");
        return 1;
    }

    if (calclLink(&image, &obj, 1) != CAL_RESULT_OK)
    {
        fprintf(stdout, "Program linking failed. Exiting.\n");
        return 1;
    }

    CALresource res[3];
    const int nx = 256;
    const int ny = 1;

    calResAllocLocal1D(&res[0], device, nx, CAL_FORMAT_FLOAT_1, 0); // for a[]
    calResAllocLocal1D(&res[1], device, nx, CAL_FORMAT_FLOAT_1, 0); // for c[]

    // Filling values in input buffer
    float* fdata = NULL;
    float* tmp;
    CALuint pitch = 0;
    int count;

    if (calResMap((CALvoid**)&fdata, &pitch, res[0], 0) == CAL_RESULT_ERROR) exit(1);
    count = 1;
    tmp = &fdata[0];
    for (int j = 0; j < nx; ++j) {
      tmp[j] = (float)(count++);
    }
    calResUnmap(res[0]);


    // Creating module using compiled image
    CALmodule module = 0;
    calModuleLoad(&module, ctx, image);

    // Defining symbols in module
    // Defining entry point for the module
    CALfunc func = 0;
    calModuleGetEntry(&func, ctx, module, "main");

    CALname name[3];
    calModuleGetName(&name[0], ctx, module, "i0");
    calModuleGetName(&name[1], ctx, module, "o0");

    // Setting input and output buffers
    // used in the kernel
    CALmem mem[3];
    for(int i = 0; i < 2; i++) {
      calCtxGetMem(&mem[i], ctx, res[i]);
      calCtxSetMem(ctx, name[i], mem[i]);
    }

    // Setting domain
    CALdomain domain = {0, 0, nx, ny};
    //-------------------------------------------------------------------------
    // Executing program and waiting for program to terminate
    //-------------------------------------------------------------------------

    // Event to check completion of the program
    CALevent e = 0;
    calCtxRunProgram(&e, ctx, func, &domain);

    // Checking whether the execution of the program is complete or not
    while (calCtxIsEventDone(ctx, e) == CAL_RESULT_PENDING);

    // Reading output from output resources
    calResMap((CALvoid**)&fdata, &pitch, res[1], 0);
    tmp = &fdata[0];
    for (int j = 0; j < 16; ++j) {
      printf("%f ", tmp[j]);
      if (j % 8 == 7) puts("");
    }
    calResUnmap(res[1]);

    //-------------------------------------------------------------------------
    // Cleaning up
    //-------------------------------------------------------------------------

    // Unloading the module
    calModuleUnload(ctx, module);

    // Freeing compiled program binary
    calclFreeImage(image);
    calclFreeObject(obj);

    // Releasing resource from context
    for(int i = 0; i < 3; i++) calCtxReleaseMem(ctx, mem[i]);

    // Deallocating resources
    for(int i = 0; i < 3; i++) calResFree(res[i]);

    // Destroying context
    calCtxDestroy(ctx);

    // Closing device
    calDeviceClose(device);

    // Shutting down CAL
    calShutdown();

    bool waitForUser = true;
    if( waitForUser )
    {
        printf("\nPress enter to exit...\n");
        getchar();
    }

    return 0;
}

